import { FormEvent, useState } from "react";
import { signIn, useSession } from "next-auth/react";
import { useRouter } from "next/router";
import { useLanguage } from "@/contexts/LanguageContext";
import Link from "next/link";

export default function LoginPage() {
  const { data: session } = useSession();
  const router = useRouter();
  const { t } = useLanguage();

  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [err, setErr] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);

  if (session?.user) {
    return (
      <div className='min-h-screen bg-sidebar flex items-center justify-center px-4'>
        <div className='w-full max-w-md rounded-2xl border border-card/60 bg-card shadow-xl p-8 text-center'>
          <div className='w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4'>
            <svg
              className='w-8 h-8 text-green-600'
              fill='none'
              stroke='currentColor'
              viewBox='0 0 24 24'
              aria-hidden
            >
              <path
                strokeLinecap='round'
                strokeLinejoin='round'
                strokeWidth={2}
                d='M5 13l4 4L19 7'
              />
            </svg>
          </div>
          <h1 className='text-2xl font-bold text-foreground mb-2'>
            Welcome Back!
          </h1>
          <p className='text-muted mb-6'>
            You're already signed in as {session.user.email}
          </p>
          <Link
            href='/'
            className='inline-block w-full rounded-xl bg-primary px-4 py-2.5 text-white font-medium hover:opacity-90 transition'
          >
            Go to Dashboard
          </Link>
        </div>
      </div>
    );
  }

  const onSubmit = async (e: FormEvent) => {
    e.preventDefault();
    setErr(null);
    setLoading(true);
    const res = await signIn("credentials", {
      redirect: false,
      email,
      password,
    });
    setLoading(false);
    if (!res || !res.ok) {
      setErr(t("login.invalidCredentials"));
      return;
    }
    router.push("/");
  };

  return (
    <div className='min-h-screen bg-sidebar relative flex items-center justify-center px-4'>
      {/* Background decoration */}
      <div
        aria-hidden
        className='pointer-events-none absolute inset-0 -z-10 overflow-hidden'
      >
        <div className='absolute -top-32 -right-24 w-80 h-80 bg-sidebar-accent/40 rounded-full blur-3xl' />
        <div className='absolute -bottom-32 -left-24 w-80 h-80 bg-primary/20 rounded-full blur-3xl' />
      </div>

      <div className='w-full max-w-md'>
        <div className='rounded-2xl border border-card/60 bg-card shadow-xl'>
          {/* Header */}
          <div className='px-8 pt-8 pb-5 text-center'>
            <div className='mx-auto mb-4 flex h-16 w-16 items-center justify-center rounded-full bg-primary/10'>
              <svg
                className='h-8 w-8 text-primary'
                fill='none'
                stroke='currentColor'
                viewBox='0 0 24 24'
                aria-hidden
              >
                <path
                  strokeLinecap='round'
                  strokeLinejoin='round'
                  strokeWidth={2}
                  d='M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.25 2.25 0 11-4.5 0 2.25 2.25 0 014.5 0z'
                />
              </svg>
            </div>
            <h1 className='text-3xl font-bold text-foreground mb-1'>
              {t("nav.hospitalPortal")}
            </h1>
            <p className='text-muted'>{t("login.signInTo")}</p>
          </div>

          {/* Form */}
          <form onSubmit={onSubmit} className='px-8 pb-8 space-y-6'>
            {err && (
              <div
                role='alert'
                aria-live='polite'
                className='bg-red-50 border border-red-200 rounded-lg p-4 flex items-start text-sm text-red-800'
              >
                <svg
                  className='w-4 h-4 mr-2 mt-0.5'
                  fill='none'
                  stroke='currentColor'
                  viewBox='0 0 24 24'
                  aria-hidden
                >
                  <path
                    strokeLinecap='round'
                    strokeLinejoin='round'
                    strokeWidth={2}
                    d='M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z'
                  />
                </svg>
                {err}
              </div>
            )}

            <div className='space-y-4'>
              <div>
                <label
                  htmlFor='email'
                  className='block text-sm font-medium text-foreground mb-2'
                >
                  {t("login.email")}
                </label>
                <input
                  id='email'
                  type='email'
                  required
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  className='w-full rounded-xl border border-input bg-background px-3.5 py-2.5 text-foreground placeholder:text-muted focus:outline-none focus:ring-2 focus:ring-primary/50'
                  placeholder={t("login.enterEmail") || "Enter your email"}
                  autoComplete='email'
                />
              </div>

              <div>
                <label
                  htmlFor='password'
                  className='block text-sm font-medium text-foreground mb-2'
                >
                  {t("login.password")}
                </label>
                <input
                  id='password'
                  type='password'
                  required
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  className='w-full rounded-xl border border-input bg-background px-3.5 py-2.5 text-foreground placeholder:text-muted focus:outline-none focus:ring-2 focus:ring-primary/50'
                  placeholder={t("login.enterPassword")}
                  autoComplete='current-password'
                />
              </div>
            </div>

            <div className='flex items-center justify-between'>
              <Link
                href='/forgot-password'
                className='text-sm text-primary hover:opacity-80'
              >
                {t("login.forgotPassword") || "Forgot password?"}
              </Link>
            </div>

            <button
              type='submit'
              disabled={loading}
              className='w-full inline-flex items-center justify-center rounded-xl bg-primary px-4 py-2.5 text-white font-medium shadow hover:opacity-90 disabled:opacity-50 disabled:cursor-not-allowed transition'
            >
              {loading ? (
                <>
                  <svg
                    className='animate-spin -ml-1 mr-2 h-5 w-5'
                    xmlns='http://www.w3.org/2000/svg'
                    fill='none'
                    viewBox='0 0 24 24'
                    aria-hidden
                  >
                    <circle
                      className='opacity-25'
                      cx='12'
                      cy='12'
                      r='10'
                      stroke='currentColor'
                      strokeWidth='4'
                    ></circle>
                    <path
                      className='opacity-75'
                      fill='currentColor'
                      d='M4 12a8 8 0 018-8V1C5.373 1 1 5.373 1 12h3z'
                    ></path>
                  </svg>
                  {t("login.signingIn")}
                </>
              ) : (
                t("login.signIn")
              )}
            </button>
          </form>

          {/* Footer Links */}
          <div className='px-8 pb-6 pt-2 border-t border-card space-y-3 text-center'>
            <p className='text-sm text-muted'>
              {t("login.dontHaveAccount")}{" "}
              <Link
                href='/signup'
                className='text-primary hover:opacity-80 font-medium'
              >
                {t("login.createOne")}
              </Link>
            </p>
            <p className='text-sm'>
              <Link
                href='/'
                className='text-muted hover:text-foreground inline-flex items-center justify-center'
              >
                <svg
                  className='w-4 h-4 mr-1'
                  fill='none'
                  stroke='currentColor'
                  viewBox='0 0 24 24'
                  aria-hidden
                >
                  <path
                    strokeLinecap='round'
                    strokeLinejoin='round'
                    strokeWidth={2}
                    d='M10 19l-7-7m0 0l7-7m-7 7h18'
                  />
                </svg>
                {t("common.back")} {t("common.to") || "to"}{" "}
                {t("common.home") || "Home"}
              </Link>
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}
